//*************************************************************************************************
//
//	Description:
//		MeshVisualisation.fx - Shader for visualizing mesh properties.
//
//	<P> Copyright (c) 2008 Blimey! Games Ltd. All rights reserved.
//
//*************************************************************************************************

#include "stddefs.fxh"

//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Transformation matrices
//

// View & Projection

#if defined( _3DSMAX_ ) || defined(_PS3_)
// Max doesn't support viewproj as an app-supplied parameter
SHARE_PARAM float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif


// Object-space into world space matrix.

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;


//
// Visualization parameter.
//

float2 uvCoordinates
<
	string UIName = "UV Coordinates";
	int coherenceType = ECT_DYNAMIC;
	bool appEdit = true;
> = { 1.0f, 1.0f };


//
// Textures
//

// False colour map.

texture falseColourTexture : TEXTURE
<
	string UIName = "False Colour Map";
	bool appEdit = true;
>;


//
// Samplers
//

// False colour map.

sampler2D falseColourMap : SAMPLER
<
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture = "falseColourTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "None";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
>
= sampler_state
{
	Texture = < falseColourTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = None;
	AddressU  = Clamp;
	AddressV  = Clamp;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};


//-----------------------------------------------------------------------
//
// Vertex Shader
//

// Input structure.

struct VSINPUT
{
	float3 position : POSITION;											// Object space position.
};


// Output structure.

struct VSOUTPUT
{
	float4 position	: POSITION;											// View-coords position.
	float2 texCoord : TEXCOORD0;										// UV coords into false colour map.
};


// Main vertex shader.

VSOUTPUT VisualiseVertexShader( VSINPUT input )
{
	VSOUTPUT output;

	// Calculate clip-space position of the vertex

#if !defined( _3DSMAX_ ) && !defined(_PS3_)
	float4x4 worldviewproj = mul( world, viewproj );
#endif
	output.position = mul( float4( input.position, 1.0f ), worldviewproj );

	// Copy the supplied UVs to vertex.

	output.texCoord = uvCoordinates;

	return output;
}


//-----------------------------------------------------------------------
//
// Fragment Shaders
//

float4 VisualiseFragmentShader( VSOUTPUT input ) : COLOR0
{
	float4 colour = tex2D( falseColourMap, input.texCoord );

	// Allow up to 10 layers of overdraw without saturation.

	return float4( colour.x, colour.y, colour.z, 0.1f );
}


//-----------------------------------------------------------------------
//
// Technique
//

technique Visualise
<
	bool supportsSpecialisedLighting = false;
  bool preservesGlobalState = true;
	string normalBehaviour	= "ERMB_RENDER";
	string normalTechnique	= "Visualise";
	int    normalDeferredID	= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = false;
		bool	AlphaBlendEnable = true;
		string SrcBlend = "SRCALPHA";
		string DestBlend = "ONE";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = SRCALPHA;
		DestBlend = ONE;
		BlendOp = ADD;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx VisualiseVertexShader();
		PixelShader = compile sce_fp_rsx VisualiseFragmentShader();
#else
		VertexShader = compile vs_3_0 VisualiseVertexShader();
		PixelShader = compile ps_3_0 VisualiseFragmentShader();
#endif
	}
}
